/*
 * Decompiled with CFR 0.152.
 */
package com.dbclient.jdbc.server;

import com.dbclient.jdbc.server.driver.DriverLoader;
import com.dbclient.jdbc.server.dto.ColumnMeta;
import com.dbclient.jdbc.server.dto.ConnectDTO;
import com.dbclient.jdbc.server.dto.QueryBO;
import com.dbclient.jdbc.server.dto.execute.ExecuteDTO;
import com.dbclient.jdbc.server.dto.execute.SQLParam;
import com.dbclient.jdbc.server.response.ExecuteResponse;
import com.dbclient.jdbc.server.util.PatternUtils;
import com.dbclient.jdbc.server.util.TypeChecker;
import com.dbclient.jdbc.server.util.ValueUtils;
import java.nio.ByteBuffer;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLSyntaxErrorException;
import java.sql.SQLXML;
import java.sql.Statement;
import java.sql.Struct;
import java.sql.Time;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import oracle.sql.REF;
import oracle.sql.TIMESTAMP;
import oracle.sql.TIMESTAMPTZ;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcExecutor {
    private static final Logger log = LoggerFactory.getLogger(JdbcExecutor.class);
    private static final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    private static final DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private Set<Statement> statements = new HashSet<Statement>();
    private String aliveSQL;
    private final ConnectDTO option;
    private final Connection connection;

    public JdbcExecutor(ConnectDTO connectDTO) {
        this.option = connectDTO;
        this.initAliveSQL();
        DriverLoader.loadDriverByDTO(connectDTO);
        Properties properties = JdbcExecutor.getConnectProperties(connectDTO);
        this.connection = DriverManager.getConnection(connectDTO.getJdbcUrl(), properties);
        if (connectDTO.isReadonly()) {
            this.connection.setReadOnly(true);
        }
    }

    private static Properties getConnectProperties(ConnectDTO connectDTO) {
        Map<String, Object> paramProperties;
        Properties properties = new Properties();
        if (!ValueUtils.isEmpty(connectDTO.getUsername())) {
            properties.put("user", connectDTO.getUsername());
        }
        if (!ValueUtils.isEmpty(connectDTO.getPassword())) {
            properties.put("password", connectDTO.getPassword());
        }
        if ((paramProperties = connectDTO.getProperties()) != null) {
            properties.putAll(paramProperties);
        }
        return properties;
    }

    public List<ExecuteResponse> executeBatch(String[] sqlList, ExecuteDTO executeDTO) {
        executeDTO.setParams(null);
        return Arrays.stream(sqlList).map(s2 -> this.execute((String)s2, executeDTO)).collect(Collectors.toList());
    }

    public boolean isOracle() {
        return this.option.getJdbcUrl().contains("oracle");
    }

    public void testAlive() {
        try {
            this.execute(this.aliveSQL, new ExecuteDTO());
        }
        catch (Exception e2) {
            if (e2 instanceof SQLSyntaxErrorException) {
                log.error(e2.getMessage(), e2);
                return;
            }
            throw new RuntimeException(e2);
        }
    }

    private void initAliveSQL() {
        this.aliveSQL = this.option.getAliveSQL();
        if (ValueUtils.isEmpty(this.aliveSQL)) {
            this.aliveSQL = "SELECT 1";
            if (this.isOracle()) {
                this.aliveSQL = this.aliveSQL + " FROM DUAL";
            }
        }
    }

    public synchronized ExecuteResponse execute(String sql, ExecuteDTO executeDTO) {
        PreparedStatement statement = this.connection.prepareStatement(sql);
        this.fillParams(statement, executeDTO.getParams());
        this.doPagination(sql, statement, executeDTO);
        boolean hasResult = statement.execute();
        if (hasResult) {
            QueryBO queryBO = this.handleResult(statement, executeDTO);
            return ExecuteResponse.builder().rows(queryBO.getRows()).columns(queryBO.getColumns()).build();
        }
        int affectedRows = statement.getUpdateCount();
        this.closeStatement(statement);
        return ExecuteResponse.builder().affectedRows(affectedRows).build();
    }

    private void doPagination(String sql, Statement statement, ExecuteDTO executeDTO) {
        String lowerSQL = sql.toLowerCase();
        boolean isQuery = PatternUtils.match(lowerSQL, "^\\s*(select|show|desc|describe|with)");
        if (!isQuery) {
            return;
        }
        Integer skipRows = executeDTO.getSkipRows();
        Integer fetchSize = executeDTO.getFetchSize();
        if (fetchSize != null && skipRows == null) {
            statement.setMaxRows(fetchSize);
            statement.setFetchSize(fetchSize);
        }
    }

    private void fillParams(PreparedStatement statement, SQLParam[] params) {
        if (params == null) {
            return;
        }
        block5: for (int i2 = 0; i2 < params.length; ++i2) {
            SQLParam param = params[i2];
            if (param == null) continue;
            switch (param.getType()) {
                case HEX: {
                    statement.setBytes(i2 + 1, ValueUtils.hexToBytes(param.getValue()));
                    continue block5;
                }
                default: {
                    statement.setObject(i2 + 1, param.getValue());
                }
            }
        }
    }

    private Statement newStatement() {
        Statement statement = this.isOracle() || this.option.isSupportForward() ? this.connection.createStatement(1004, 1007) : this.connection.createStatement();
        this.statements.add(statement);
        return statement;
    }

    private void closeStatement(Statement statement) throws SQLException {
        try {
            statement.close();
        }
        finally {
            this.statements.remove(statement);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public QueryBO handleResult(Statement statement, ExecuteDTO executeDTO) {
        int i2;
        ResultSet rs = statement.getResultSet();
        Integer skipRows = executeDTO.getSkipRows();
        Integer fetchSize = executeDTO.getFetchSize();
        log.info("Query completed, start fetching data...");
        ResultSetMetaData metaData = rs.getMetaData();
        int columnCount = metaData.getColumnCount();
        ArrayList<ColumnMeta> columns = new ArrayList<ColumnMeta>(columnCount);
        for (i2 = 0; i2 < columnCount; ++i2) {
            int i1 = i2 + 1;
            columns.add(new ColumnMeta(metaData.getColumnLabel(i1), metaData.getColumnTypeName(i1), this.getTableName(metaData, i1)));
        }
        if (skipRows != null) {
            if (this.isOracle() || this.option.isSupportForward()) {
                if (fetchSize != null) {
                    rs.setFetchSize(fetchSize);
                }
                rs.absolute(skipRows);
            } else {
                for (i2 = 0; i2 < skipRows; ++i2) {
                    rs.next();
                }
            }
        }
        ArrayList<List<Object>> rows = new ArrayList<List<Object>>();
        try {
            for (int fetchedCount = 0; rs.next() && (fetchSize == null || fetchedCount < fetchSize); ++fetchedCount) {
                ArrayList<Object> row = new ArrayList<Object>(columnCount);
                for (int i3 = 1; i3 <= columnCount; ++i3) {
                    try {
                        row.add(this.getColumnValue(rs, i3));
                        continue;
                    }
                    catch (SQLException e2) {
                        log.error(e2.getMessage(), e2);
                    }
                }
                rows.add(row);
            }
        }
        finally {
            this.closeStatement(statement);
        }
        log.info("Data fetching completed!");
        return new QueryBO(rows, columns);
    }

    private String getTableName(ResultSetMetaData metaData, int i1) throws SQLException {
        if (this.option.getJdbcUrl().contains("hive") || this.option.isNoTableName()) {
            return null;
        }
        try {
            return metaData.getTableName(i1);
        }
        catch (Exception e2) {
            log.error(e2.getMessage(), e2);
            return null;
        }
    }

    private Object getColumnValue(ResultSet rs, int i2) throws SQLException {
        Object object = rs.getObject(i2);
        return JdbcExecutor.convertValue(object);
    }

    private static Object convertValue(Object object) {
        if (object == null) {
            return null;
        }
        if (TypeChecker.isPrimary(object.getClass())) {
            return object;
        }
        if (object instanceof ByteBuffer) {
            return ValueUtils.bytesToHex(((ByteBuffer)object).array());
        }
        if (object instanceof Clob) {
            Clob clob = (Clob)object;
            return clob.getSubString(1L, (int)clob.length());
        }
        if (object instanceof Blob) {
            byte[] bytes = ((Blob)object).getBytes(1L, (int)((Blob)object).length());
            return ValueUtils.bytesToHex(bytes);
        }
        if (object instanceof Date || object instanceof Time) {
            return object.toString();
        }
        if (object instanceof Timestamp) {
            return ((Timestamp)object).toLocalDateTime().format(dateTimeFormatter);
        }
        if (object instanceof TIMESTAMP) {
            return ((TIMESTAMP)object).toLocalDateTime().format(dateTimeFormatter);
        }
        if (object instanceof TIMESTAMPTZ) {
            return ((TIMESTAMPTZ)object).toLocalDateTime().format(dateTimeFormatter);
        }
        if (object instanceof java.util.Date) {
            return dateFormat.format(object);
        }
        if (object instanceof Array) {
            return Arrays.stream((Object[])((Array)object).getArray()).map(JdbcExecutor::convertValue).toArray(Object[]::new);
        }
        if (object.getClass() == byte[].class) {
            return ValueUtils.bytesToHex((byte[])object);
        }
        if (object.getClass() == Object[].class) {
            return Arrays.stream((Object[])object).map(JdbcExecutor::convertValue).toArray(Object[]::new);
        }
        if (object instanceof SQLXML) {
            return ((SQLXML)object).getString();
        }
        if (object instanceof Struct) {
            return JdbcExecutor.convertValue(((Struct)object).getAttributes());
        }
        if (object instanceof Ref) {
            return ((Ref)object).getBaseTypeName() + " -> {" + Arrays.stream((Object[])JdbcExecutor.convertValue(((Struct)((REF)object).getObject()).getAttributes())).map(String::valueOf).collect(Collectors.joining(", ")) + "}";
        }
        return object.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void cancel() {
        for (Statement statement : this.statements) {
            try {
                statement.cancel();
            }
            catch (SQLException e2) {
                log.error(e2.getMessage(), e2);
            }
            finally {
                statement.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() throws Exception {
        for (Statement statement : this.statements) {
            try {
                statement.cancel();
            }
            catch (SQLException e2) {
                log.error(e2.getMessage(), e2);
            }
            finally {
                statement.close();
            }
        }
        if (this.connection != null) {
            this.connection.close();
        }
    }

    public Connection getConnection() {
        return this.connection;
    }
}

