"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebviewWizardPage = void 0;
const WizardPage_1 = require("./WizardPage");
const WebviewWizard_1 = require("./WebviewWizard");
const StandardWizardPageRenderer_1 = require("./StandardWizardPageRenderer");
class WebviewWizardPage extends WizardPage_1.WizardPage {
    constructor(pageDefinition, wizardDefinition) {
        super(pageDefinition.id, pageDefinition.title, pageDefinition.description);
        this.fieldStateCache = new Map();
        this.wizardDefinition = wizardDefinition;
        this.pageDefinition = pageDefinition;
        super.setFocusedField(this.findFocusedField());
        this.initializeStateCache();
    }
    initializeStateCache() {
        for (let field of this.pageDefinition.fields) {
            if (this.isWizardPageSectionDefinition(field)) {
                for (let childField of field.childFields) {
                    if (childField.initialState) {
                        this.fieldStateCache.set(childField.id, childField.initialState);
                    }
                }
            }
            else if (field.initialState) {
                this.fieldStateCache.set(field.id, field.initialState);
            }
        }
    }
    isWizardPageSectionDefinition(def) {
        return def.childFields !== undefined;
    }
    findFocusedField() {
        if (this.pageDefinition.fields.length > 0) {
            const field = this.pageDefinition.fields.find(field => WebviewWizard_1.isWizardPageFieldDefinition(field) && field.focus === true);
            return field ? field.id : this.pageDefinition.fields[0].id;
        }
    }
    getPageDefinition() {
        return this.pageDefinition;
    }
    /**
     * Validate the wizard page by updating the page complete flag only.
     *
     * @param parameters the current parameters.
     * @param previousParameters the previous parameters.
     */
    validateAndUpdatePageComplete(parameters, previousParameters) {
        const hasError = this.getValidationStatus(parameters, previousParameters);
        this.setPageComplete(!hasError);
    }
    getValidationStatus(parameters, previousParameters) {
        const resp = this.doValidate(parameters, previousParameters);
        return (resp && resp.items && resp.items.some(item => item.severity === WebviewWizard_1.SEVERITY.ERROR)) || false;
    }
    doValidate(parameters, previousParameters) {
        if (this.pageDefinition.validator) {
            return this.pageDefinition.validator.call(null, parameters, previousParameters);
        }
    }
    /**
     * Validate the wizard page by updating the page complete flag and collect HTML content to update (as Template) for :
     *  - validation messages.
     *  - widget to redraw according a state (not enabled, not visible, etc)
     *
     * @param parameters the current parameters.
     * @param previousParameters the previous parameters.
     *
     * @returns Template collection
     */
    getValidationTemplates(parameters, previousParameters) {
        this.setPageComplete(true);
        return this.validate(parameters, previousParameters);
    }
    severityToImage(severity) {
        switch (severity) {
            case WebviewWizard_1.SEVERITY.ERROR:
                return '<i class="icon icon__error"></i>';
            case WebviewWizard_1.SEVERITY.WARN:
                return '<i class="icon icon__warn"></i>';
            case WebviewWizard_1.SEVERITY.INFO:
                return '<i class="icon icon__info"></i>';
            default:
                return "";
        }
    }
    severityToCSSClass(severity) {
        switch (severity) {
            case WebviewWizard_1.SEVERITY.ERROR:
                return 'error-message';
            case WebviewWizard_1.SEVERITY.WARN:
                return 'warn-message';
            case WebviewWizard_1.SEVERITY.INFO:
                return 'info-message';
            default:
                return "";
        }
    }
    validate(parameters, previousParameters) {
        let templates = [];
        const resp = this.doValidate(parameters, previousParameters);
        if (resp) {
            // If validation has returned any widgets to refresh, we should do that now
            if (resp.fieldRefresh) {
                for (let [key, value] of resp.fieldRefresh) {
                    let def = this.findFieldDefinition(key);
                    let currentState = this.fieldStateCache.get(key);
                    if (currentState === undefined || currentState === null) {
                        currentState = {};
                        this.fieldStateCache.set(key, currentState);
                    }
                    let stateChanged = false;
                    if (value.hasOwnProperty("enabled")) {
                        stateChanged = currentState.enabled !== value.enabled;
                        currentState.enabled = value.enabled;
                    }
                    if (value.hasOwnProperty("visible")) {
                        stateChanged = stateChanged || currentState.visible !== value.visible;
                        currentState.visible = value.visible;
                    }
                    if (def !== null && stateChanged) {
                        let str = this.getRenderer().oneFieldAsString(def, parameters);
                        templates = templates.concat({ id: key + "Field", content: str });
                    }
                }
            }
            for (let item of resp.items) {
                const { severity, template } = item;
                // Allow users to just put the failed field id here. We add Validation
                if (!template.id.endsWith("Validation")) {
                    template.id = template.id + "Validation";
                }
                if (severity === WebviewWizard_1.SEVERITY.ERROR) {
                    this.setPageComplete(false);
                }
                const img = this.severityToImage(severity);
                const cssClass = this.severityToCSSClass(severity);
                template.content = `<div class="${cssClass}">${img}${template.content || "&nbsp;"}</div>`;
                templates = templates.concat(template);
            }
        }
        // All the official ones were added.
        // Now lets clear out all the empty ones
        for (let key of this.pageDefinition.fields) {
            if (WebviewWizard_1.isWizardPageSectionDefinition(key)) {
                for (let key2 of key.childFields) {
                    if (!this.containsTemplate(key2.id, templates)) {
                        templates.push({ id: key2.id + "Validation", content: "&nbsp;" });
                    }
                }
            }
            else if (WebviewWizard_1.isWizardPageFieldDefinition(key)) {
                if (!this.containsTemplate(key.id, templates)) {
                    templates.push({ id: key.id + "Validation", content: "&nbsp;" });
                }
            }
        }
        return templates;
    }
    findFieldDefinition(id) {
        for (let i of this.pageDefinition.fields) {
            if (WebviewWizard_1.isWizardPageSectionDefinition(i)) {
                for (let j of i.childFields) {
                    if (j.id == id)
                        return j;
                }
            }
            else if (WebviewWizard_1.isWizardPageFieldDefinition(i)) {
                if (i.id == id)
                    return i;
            }
        }
        return null;
    }
    containsTemplate(id, templates) {
        for (let template of templates) {
            if (template.id === id || template.id === (id + 'Validation')) {
                return true;
            }
        }
        return false;
    }
    getRenderer() {
        if (this.wizardDefinition && this.wizardDefinition.renderer) {
            return this.wizardDefinition.renderer;
        }
        return new StandardWizardPageRenderer_1.StandardWizardPageRenderer(this.fieldStateCache);
    }
    getContentAsHTML(data) {
        return this.getRenderer().getContentAsHTML(this.pageDefinition, data);
    }
}
exports.WebviewWizardPage = WebviewWizardPage;
//# sourceMappingURL=WebviewWizardPage.js.map