"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateNonce = exports.StorageServiceImpl = void 0;
/*-----------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat, Inc. All rights reserved.
 *  Licensed under the EPL v2.0 License. See LICENSE file in the project root for license information.
 *-----------------------------------------------------------------------------------------------*/
const path_1 = __importDefault(require("path"));
const vscode_1 = require("vscode");
const fsUtil_1 = require("./util/fsUtil");
class StorageServiceImpl {
    constructor(storagePath) {
        this.storagePath = storagePath;
    }
    delay(ms) {
        return __awaiter(this, void 0, void 0, function* () {
            yield new Promise(resolve => setTimeout(resolve, ms));
        });
    }
    /**
     * Run a given runnable while ensuring only 1 client can write to the data store at a time.
     * I am not convinced the lock mechanism is perfect.
     * @returns boolean - whether vscode is in a new session vs what the data store thought but only when a change to the model occurs. False otherwise
     */
    runWithLock(runnable) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.ensureStorageFolderExists();
            const acquired = yield this.acquireLock(15);
            if (!acquired) {
                return Promise.reject("Unable to acquire a lock to modify the recommendation model.");
            }
            try {
                const model = yield this.loadOrDefaultRecommendationModel();
                const now = Date.now();
                let newSession = false;
                if (vscode_1.env.sessionId !== model.sessionId) {
                    model.sessionId = vscode_1.env.sessionId;
                    model.sessionTimestamp = now;
                    model.timelocked = [];
                    newSession = true;
                }
                model.lastUpdated = now;
                const model2 = yield runnable(model);
                if (model2) {
                    model.lastUpdated = Date.now();
                    yield this.save(model2);
                }
                return newSession;
            }
            finally {
                this.unlock();
            }
        });
    }
    acquireLock(seconds) {
        return __awaiter(this, void 0, void 0, function* () {
            const endTime = Date.now() + (1000 * seconds);
            while (Date.now() <= endTime) {
                const ret = yield this.acquireLockOneTry(endTime);
                if (ret) {
                    return true;
                }
            }
            return false;
        });
    }
    acquireLockOneTry(endTime) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.deleteOldLockFile();
            const file = this.resolvePath(StorageServiceImpl.LOCK_FILENAME);
            while ((yield (0, fsUtil_1.exists)(file)) && Date.now() < endTime) {
                yield this.delay(25);
            }
            if (Date.now() >= endTime) {
                // Took too long
                return false;
            }
            // We have a chance to write a lock. Let's try it. 
            const nonce = (0, exports.generateNonce)();
            const contents = Date.now() + "\n" + nonce;
            yield this.writeToFile(file, contents);
            // Give some time for someone else to try to write to the file / race condition
            yield this.delay(100);
            // verify we are still the nonce in the file
            const nonce2 = yield this.getNonceFromLockFile();
            if (nonce2 && nonce2 === nonce) {
                return true;
            }
            return false;
        });
    }
    getTimeFromLockFile() {
        return __awaiter(this, void 0, void 0, function* () {
            const existsVar = yield (0, fsUtil_1.exists)(this.resolvePath(StorageServiceImpl.LOCK_FILENAME));
            if (existsVar) {
                const s = yield this.readFromFileOrUndefined(StorageServiceImpl.LOCK_FILENAME);
                if (s) {
                    const lines = s.split("\n");
                    if (lines && lines.length === 2 && lines[0]) {
                        return parseInt(lines[0]);
                    }
                }
            }
            return undefined;
        });
    }
    getNonceFromLockFile() {
        return __awaiter(this, void 0, void 0, function* () {
            const existsVar = yield (0, fsUtil_1.exists)(this.resolvePath(StorageServiceImpl.LOCK_FILENAME));
            if (existsVar) {
                const s = yield this.readFromFileOrUndefined(StorageServiceImpl.LOCK_FILENAME);
                if (s) {
                    const lines = s.split("\n");
                    if (lines && lines.length === 2 && lines[0]) {
                        return lines[1];
                    }
                }
            }
            return undefined;
        });
    }
    deleteOldLockFile() {
        return __awaiter(this, void 0, void 0, function* () {
            const tooOld = Date.now() - (1000 * 30); // 30 seconds lock is too long, man
            const ts = yield this.getTimeFromLockFile();
            if (ts) {
                if (ts < tooOld) {
                    this.unlock();
                }
            }
        });
    }
    unlock() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const file = this.resolvePath(StorageServiceImpl.LOCK_FILENAME);
                yield (0, fsUtil_1.deleteFile)(file);
            }
            catch (err) {
                // File did not exist probably, ignore
            }
        });
    }
    readRecommendationModel() {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.ensureStorageFolderExists();
            return this.loadRecommendationModel();
        });
    }
    loadRecommendationModel() {
        return __awaiter(this, void 0, void 0, function* () {
            const json = yield this.readFromFileOrUndefined(StorageServiceImpl.PERSISTENCE_FILENAME);
            if (json) {
                return JSON.parse(json);
            }
            return undefined;
        });
    }
    loadOrDefaultRecommendationModel() {
        return __awaiter(this, void 0, void 0, function* () {
            const def = {
                lastUpdated: Date.now(),
                sessionId: "",
                sessionTimestamp: Date.now(),
                recommendations: [],
                timelocked: []
            };
            let ret = yield this.loadRecommendationModel();
            if (!ret) {
                return def;
            }
            return ret;
        });
    }
    writeKey(key, contents) {
        return __awaiter(this, void 0, void 0, function* () {
            // Add a prefix so no one can overwrite the core model or lock file
            const resolved = this.resolvePath("rec_" + key);
            yield (0, fsUtil_1.writeFile)(resolved, contents);
            return resolved;
        });
    }
    resolvePath(filename) {
        return path_1.default.resolve(this.storagePath, filename);
    }
    save(model) {
        return __awaiter(this, void 0, void 0, function* () {
            const json = JSON.stringify(model);
            yield this.writeToFile(StorageServiceImpl.PERSISTENCE_FILENAME, json);
        });
    }
    readFromFile(filename) {
        return __awaiter(this, void 0, void 0, function* () {
            const filePath = this.resolvePath(filename);
            return yield (0, fsUtil_1.readFile)(filePath);
        });
    }
    readFromFileOrUndefined(filename) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                return yield this.readFromFile(filename);
            }
            catch (err) {
                return undefined;
            }
        });
    }
    writeToFile(filename, value) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, fsUtil_1.writeFile)(this.resolvePath(filename), value);
        });
    }
    ensureStorageFolderExists() {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, fsUtil_1.mkdir)(this.storagePath);
        });
    }
}
exports.StorageServiceImpl = StorageServiceImpl;
StorageServiceImpl.PERSISTENCE_FILENAME = 'extension-recommender.model.json';
StorageServiceImpl.LOCK_FILENAME = 'extension-recommender.lock';
const generateNonce = () => {
    let text = "";
    const possible = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";
    for (let i = 0; i < 32; i++) {
        text += possible.charAt(Math.floor(Math.random() * possible.length));
    }
    return text;
};
exports.generateNonce = generateNonce;
//# sourceMappingURL=storageServiceImpl.js.map