"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.enhance = exports.IGNORED_USERS = void 0;
/**
 * Enhances a `TelemetryEvent` by injecting environmental data to its properties and context
 *
 * See segment.com fields: https://segment.com/docs/connections/spec/common/#integrations
 * {
  "anonymousId": "507f191e810c19729de860ea",
  "context": {
    "active": true,
    "app": {
      "name": "InitechGlobal",
      "version": "545",
      "build": "3.0.1.545",
      "namespace": "com.production.segment"
    },
    "campaign": {
      "name": "TPS Innovation Newsletter",
      "source": "Newsletter",
      "medium": "email",
      "term": "tps reports",
      "content": "image link"
    },
    "device": {
      "id": "B5372DB0-C21E-11E4-8DFC-AA07A5B093DB",
      "advertisingId": "7A3CBEA0-BDF5-11E4-8DFC-AA07A5B093DB",
      "adTrackingEnabled": true,
      "manufacturer": "Apple",
      "model": "iPhone7,2",
      "name": "maguro",
      "type": "ios",
      "token": "ff15bc0c20c4aa6cd50854ff165fd265c838e5405bfeb9571066395b8c9da449"
    },
    "ip": "8.8.8.8",
    "library": {
      "name": "analytics.js",
      "version": "2.11.1"
    },
    "locale": "en-US",
    "location": {
      "city": "San Francisco",
      "country": "United States",
      "latitude": 40.2964197,
      "longitude": -76.9411617,
      "speed": 0
    },
    "network": {
      "bluetooth": false,
      "carrier": "T-Mobile US",
      "cellular": true,
      "wifi": false
    },
    "os": {
      "name": "iPhone OS",
      "version": "8.1.3"
    },
    "page": {
      "path": "/academy/",
      "referrer": "",
      "search": "",
      "title": "Analytics Academy",
      "url": "https://segment.com/academy/"
    },
    "referrer": {
      "id": "ABCD582CDEFFFF01919",
      "type": "dataxu"
    },
    "screen": {
      "width": 320,
      "height": 568,
      "density": 2
    },
    "groupId": "12345",
    "timezone": "Europe/Amsterdam",
    "userAgent": "Mozilla/5.0 (iPhone; CPU iPhone OS 9_1 like Mac OS X) AppleWebKit/601.1.46 (KHTML, like Gecko) Version/9.0 Mobile/13B143 Safari/601.1"
  },
  "integrations": {
    "All": true,
    "Mixpanel": false,
    "Salesforce": false
  },
  "messageId": "022bb90c-bbac-11e4-8dfc-aa07a5b093db",
  "receivedAt": "2015-12-10T04:08:31.909Z",
  "sentAt": "2015-12-10T04:08:31.581Z",
  "timestamp": "2015-12-10T04:08:31.905Z",
  "type": "track",
  "userId": "97980cfea0067",
  "version": 2
}
 *
 * @param event the event to enhance
 * @param environment the environment data to inject the event with
 */
exports.IGNORED_USERS = ['user', 'gitpod', 'theia'];
function enhance(event, environment) {
    //Inject Client name and version,  Extension id and version, and timezone to the event properties
    const properties = event.properties ? sanitize(event.properties, environment) : {};
    if (!(event.type) || event.type == 'track') {
        properties.extension_name = environment.extension.name;
        properties.extension_version = environment.extension.version;
        properties.app_name = environment.application.name;
        properties.app_version = environment.application.version;
        if (environment.application.uiKind) {
            properties.app_kind = environment.application.uiKind;
        }
        if (environment.application.remote !== undefined) {
            properties.app_remote = environment.application.remote;
        }
    }
    const traits = event.traits ? sanitize(event.traits, environment) : {};
    if (event.type == 'identify') {
        //All those traits should be handled by Woopra in the context block, but are not. Meh.
        traits.timezone = environment.timezone;
        traits.os_name = environment.platform.name;
        traits.os_version = environment.platform.version;
        traits.os_distribution = environment.platform.distribution;
        traits.locale = environment.locale;
    }
    //Inject Plateform specific data in segment's context, so it can be recognized by the end destination
    // XXX Currently, Woopra ignores app, os, locale and timezone
    const context = event.context ? event.context : {};
    context.ip = '0.0.0.0';
    context.app = {
        name: environment.application.name,
        version: environment.application.version
    };
    context.os = {
        name: environment.platform.name,
        version: environment.platform.version,
    };
    context.locale = environment.locale;
    context.location = {
        // This is inacurate in some cases (user uses a different locale than from his actual country), 
        // but still provides an interesting metric in most cases. 
        country: environment.country
    };
    context.timezone = environment.timezone;
    const enhancedEvent = {
        name: event.name,
        type: event.type,
        properties: properties,
        measures: event.measures,
        traits: traits,
        context: context
    };
    return enhancedEvent;
}
exports.enhance = enhance;
function sanitize(properties, environment) {
    const sanitized = {};
    let usernameRegexp;
    if (environment.username && environment.username.length > 3 && !exports.IGNORED_USERS.includes(environment.username)) {
        usernameRegexp = new RegExp(environment.username, 'g');
    }
    for (const p in properties) {
        const rawProperty = properties[p];
        if (!rawProperty || !usernameRegexp || isNonStringPrimitive(rawProperty)) {
            sanitized[p] = rawProperty;
            continue;
        }
        const isObj = isObject(rawProperty);
        let sanitizedProperty = isObj ? JSON.stringify(rawProperty) : rawProperty;
        //TODO implement less aggressive path stripping
        //sanitizedProperty = stripPaths(sanitizedProperty);
        sanitizedProperty = sanitizedProperty.replace(usernameRegexp, '_username_');
        if (isObj) {
            //let's try to deserialize into a sanitized object
            try {
                sanitizedProperty = JSON.parse(sanitizedProperty);
            }
            catch (e) {
                //We messed up, we'll return the sanitized string instead
            }
        }
        sanitized[p] = sanitizedProperty;
    }
    return sanitized;
}
// Extremely naive regexp to capture anything that looks like a path
// It will certainly either be too aggressive or insufficient, depending 
// on the case. eg. doesn't handle paths containing spaces properly 
const naivePattern = /(([a-zA-Z](:)\\[^\s]*)|([^\s]*\/[^\s]*))/g;
function stripPaths(rawProperty) {
    return rawProperty.replace(naivePattern, 'anonymized/path');
}
function isObject(test) {
    return test === Object(test);
}
function isNonStringPrimitive(test) {
    return typeof test !== "string" && !(test instanceof String) && !isObject(test);
}
//# sourceMappingURL=events.js.map