# SQL Server 2022 Express Installation Script - Clean Version
# Autor: T. Balcke und Claude Sonnet 4 - August 2025

param(
    [Parameter(Mandatory = $true)]
    [string]$SAPassword,
    
    [string]$InstanceName = "APOCDNEXTGEN",
    
    [string]$InstallPath = "C:\Program Files\Microsoft SQL Server",
    
    [string]$LocalSetupPath = ".\2022",
    
    [switch]$ForceEncryption,
    
    [string]$CertificateThumbprint,
    
    [switch]$Silent = $false
)

# Globale Variablen
$global:LogFile = Join-Path $PSScriptRoot "SqlServerInstall.log"
$global:Silent = $Silent

# Logging-Funktion
function Write-Log {
    param(
        [string]$Message,
        [string]$Level = "INFO"
    )
    
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logEntry = "[$timestamp] [$Level] $Message"
    
    Add-Content -Path $global:LogFile -Value $logEntry -Encoding UTF8
    
    if (-not $global:Silent) {
        switch ($Level) {
            "ERROR" { Write-Host $logEntry -ForegroundColor Red }
            "WARNING" { Write-Host $logEntry -ForegroundColor Yellow }
            "SUCCESS" { Write-Host $logEntry -ForegroundColor Green }
            "INFO" { Write-Host $logEntry -ForegroundColor Cyan }
            default { Write-Host $logEntry }
        }
    }
}

function Test-AdminRights {
    try {
        $currentUser = [Security.Principal.WindowsIdentity]::GetCurrent()
        $principal = New-Object Security.Principal.WindowsPrincipal($currentUser)
        return $principal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
    }
    catch {
        return $false
    }
}

function Test-SqlServerInstalled {
    param([string]$InstanceName)
    
    try {
        $regPath = "HKLM:\SOFTWARE\Microsoft\Microsoft SQL Server\Instance Names\SQL"
        if (Test-Path $regPath) {
            $instances = Get-ItemProperty -Path $regPath -ErrorAction SilentlyContinue
            if ($instances -and $instances.PSObject.Properties.Name -contains $InstanceName) {
                return $true
            }
        }
        return $false
    }
    catch {
        Write-Log "Fehler beim Pruefen der SQL Server Installation: $_" "ERROR"
        return $false
    }
}

function Find-SqlServerSetup {
    param([string]$LocalSetupPath)
    
    $possiblePaths = @(
        (Join-Path $LocalSetupPath "setup.exe"),
        (Join-Path $LocalSetupPath "SQL2022\setup.exe"),
        (Join-Path $LocalSetupPath "SQLEXPR\setup.exe"),
        (Join-Path $LocalSetupPath "SQLEXPRADV\setup.exe")
    )
    
    foreach ($path in $possiblePaths) {
        $fullPath = Resolve-Path $path -ErrorAction SilentlyContinue
        if ($fullPath -and (Test-Path $fullPath)) {
            Write-Log "Setup gefunden: $fullPath" "SUCCESS"
            return $fullPath.Path
        }
    }
    
    Write-Log "Setup.exe nicht gefunden in: $LocalSetupPath" "ERROR"
    Write-Log "Gesuchte Pfade:" "INFO"
    foreach ($path in $possiblePaths) {
        Write-Log "  - $path" "INFO"
    }
    
    return $null
}

function Test-PasswordComplexity {
    param([string]$Password)
    
    if ($Password.Length -lt 8) {
        Write-Log "Passwort muss mindestens 8 Zeichen lang sein" "ERROR"
        return $false
    }
    
    $hasUpper = $Password -cmatch '[A-Z]'
    $hasLower = $Password -cmatch '[a-z]'
    $hasDigit = $Password -cmatch '\d'
    $hasSpecial = $Password -cmatch '[^a-zA-Z0-9]'
    
    if (-not ($hasUpper -and $hasLower -and $hasDigit -and $hasSpecial)) {
        Write-Log "Passwort muss Gross-, Kleinbuchstaben, Ziffern und Sonderzeichen enthalten" "ERROR"
        return $false
    }
    
    return $true
}

function Install-SqlServer {
    param(
        [string]$SetupPath,
        [string]$InstanceName,
        [string]$SAPassword,
        [string]$InstallPath
    )
    
    try {
        Write-Log "=== SQL Server 2022 Express Installation gestartet ===" "INFO"
        Write-Log "Instanz: $InstanceName" "INFO"
        Write-Log "Installationspfad: $InstallPath" "INFO"
        Write-Log "Setup: $SetupPath" "INFO"
        
        $setupArgs = @(
            "/ACTION=Install",
            "/FEATURES=SQLEngine",
            "/INSTANCENAME=$InstanceName",
            "/INSTANCEDIR=`"$InstallPath`"",
            "/SQLSVCACCOUNT=`"NT SERVICE\MSSQL`$$InstanceName`"",
            "/SQLSVCSTARTUPTYPE=Automatic",
            "/SQLSYSADMINACCOUNTS=`"BUILTIN\Administrators`"",
            "/SECURITYMODE=SQL",
            "/SAPWD=`"$SAPassword`"",
            "/TCPENABLED=1",
            "/NPENABLED=0",
            "/SKIPRULES=RebootRequiredCheck",
            "/QUIET",
            "/INDICATEPROGRESS",
            "/SUPPRESSPRIVACYSTATEMENTNOTICE",
            "/IACCEPTSQLSERVERLICENSETERMS",
            "/SQLCOLLATION=SQL_Latin1_General_CP1_CI_AS",
            "/INSTALLSHAREDDIR=`"C:\Program Files\Microsoft SQL Server`"",
            "/INSTALLSHAREDWOWDIR=`"C:\Program Files (x86)\Microsoft SQL Server`"",
            "/INSTALLSQLDATADIR=`"$InstallPath`"",
            "/SQLUSERDBDIR=`"$InstallPath\MSSQL16.$InstanceName\MSSQL\Data`"",
            "/SQLUSERDBLOGDIR=`"$InstallPath\MSSQL16.$InstanceName\MSSQL\Data`"",
            "/SQLBACKUPDIR=`"$InstallPath\MSSQL16.$InstanceName\MSSQL\Backup`"",
            "/SQLTEMPDBDIR=`"$InstallPath\MSSQL16.$InstanceName\MSSQL\Data`"",
            "/SQLTEMPDBLOGDIR=`"$InstallPath\MSSQL16.$InstanceName\MSSQL\Data`"",
            "/ADDCURRENTUSERASSQLADMIN=True",
            "/UPDATEENABLED=False",
            "/ERRORREPORTING=False",
            "/SQMREPORTING=False"
        )
        
        Write-Log "Starte Setup-Prozess..." "INFO"
        Write-Log "Setup-Argumente:" "INFO"
        foreach ($arg in $setupArgs) {
            $logArg = $arg -replace '(SAPWD=")[^"]*(")', '$1***$2'
            Write-Log "  $logArg" "INFO"
        }
        
        $argumentList = $setupArgs -join " "
        
        Write-Log "Fuehre aus: `"$SetupPath`" $argumentList" "INFO"
        
        $process = Start-Process -FilePath $SetupPath -ArgumentList $argumentList -Wait -PassThru -NoNewWindow
        
        $exitCode = $process.ExitCode
        Write-Log "Setup beendet mit Exit Code: $exitCode" "INFO"
        
        switch ($exitCode) {
            0 {
                Write-Log "=== Installation erfolgreich abgeschlossen ===" "SUCCESS"
                return $true
            }
            3010 {
                Write-Log "=== Installation erfolgreich - Neustart empfohlen ===" "SUCCESS"
                Write-Log "HINWEIS: Ein Systemneustart wird empfohlen" "WARNING"
                return $true
            }
            default {
                Write-Log "=== Installation fehlgeschlagen ===" "ERROR"
                Write-Log "Exit Code: $exitCode" "ERROR"
                
                $setupLogs = Get-ChildItem -Path $env:TEMP -Filter "*SQL*Setup*.log" -ErrorAction SilentlyContinue
                if ($setupLogs) {
                    Write-Log "Setup-Log-Dateien gefunden:" "INFO"
                    foreach ($log in ($setupLogs | Sort-Object LastWriteTime -Descending | Select-Object -First 3)) {
                        Write-Log "  - $($log.FullName)" "INFO"
                        
                        try {
                            $logContent = Get-Content $log.FullName -Tail 20 -ErrorAction SilentlyContinue
                            $errors = $logContent | Where-Object { $_ -match "error|failed|exception" }
                            foreach ($error in ($errors | Select-Object -First 5)) {
                                Write-Log "    ERROR: $error" "ERROR"
                            }
                        }
                        catch {}
                    }
                }
                
                return $false
            }
        }
    }
    catch {
        Write-Log "Exception waehrend Installation: $_" "ERROR"
        return $false
    }
}

function Test-InstallationSuccess {
    param([string]$InstanceName)
    
    try {
        Write-Log "Validiere Installation..." "INFO"
        
        if (-not (Test-SqlServerInstalled $InstanceName)) {
            Write-Log "Registry-Eintrag fuer Instanz nicht gefunden" "ERROR"
            return $false
        }
        
        $serviceName = "MSSQL`$$InstanceName"
        $service = Get-Service -Name $serviceName -ErrorAction SilentlyContinue
        
        if (-not $service) {
            Write-Log "SQL Server Service '$serviceName' nicht gefunden" "ERROR"
            return $false
        }
        
        Write-Log "Service Status: $($service.Status)" "INFO"
        
        if ($service.Status -ne 'Running') {
            Write-Log "Starte SQL Server Service..." "INFO"
            try {
                Start-Service -Name $serviceName
                Start-Sleep -Seconds 10
                $service.Refresh()
                
                if ($service.Status -eq 'Running') {
                    Write-Log "Service erfolgreich gestartet" "SUCCESS"
                }
                else {
                    Write-Log "Service konnte nicht gestartet werden: $($service.Status)" "ERROR"
                    return $false
                }
            }
            catch {
                Write-Log "Fehler beim Starten des Services: $_" "ERROR"
                return $false
            }
        }
        
        Write-Log "Installation erfolgreich validiert" "SUCCESS"
        return $true
    }
    catch {
        Write-Log "Fehler bei Installation-Validierung: $_" "ERROR"
        return $false
    }
}

function Get-ConnectionString {
    param(
        [string]$InstanceName,
        [bool]$UseEncryption = $true
    )
    
    $server = ".\$InstanceName"
    $connectionString = "Server=$server;User Id=sa;Password=***;Connection Timeout=30;Command Timeout=30;Application Name=SQL Server Install Verification;"
    
    if ($UseEncryption) {
        $connectionString += "Encrypt=true;TrustServerCertificate=true;"
    }
    else {
        $connectionString += "Encrypt=false;"
    }
    
    return $connectionString
}

# Hauptprogramm
try {
    Write-Log "=== SQL Server 2022 Express Installation Script gestartet ===" "INFO"
    Write-Log "PowerShell Version: $($PSVersionTable.PSVersion)" "INFO"
    
    try {
        $osInfo = Get-CimInstance Win32_OperatingSystem | Select-Object -ExpandProperty Caption
        Write-Log "Betriebssystem: $osInfo" "INFO"
    }
    catch {
        Write-Log "Betriebssystem: Unbekannt" "INFO"
    }
    
    Write-Log "Ausfuehrungszeit: $(Get-Date)" "INFO"
    Write-Log "" "INFO"
    
    Write-Log "Parameter-Validierung..." "INFO"
    Write-Log "Instanz: $InstanceName" "INFO"
    Write-Log "Installationspfad: $InstallPath" "INFO"
    Write-Log "Setup-Pfad: $LocalSetupPath" "INFO"
    
    if (-not (Test-AdminRights)) {
        Write-Log "FEHLER: Skript muss als Administrator ausgefuehrt werden!" "ERROR"
        exit 1
    }
    Write-Log "Administrator-Rechte bestaetigt" "SUCCESS"
    
    if (-not (Test-PasswordComplexity $SAPassword)) {
        Write-Log "FEHLER: SA-Passwort erfuellt nicht die Komplexitaetsanforderungen!" "ERROR"
        exit 1
    }
    Write-Log "SA-Passwort-Komplexitaet bestaetigt" "SUCCESS"
    
    if (Test-SqlServerInstalled $InstanceName) {
        Write-Log "WARNUNG: SQL Server Instanz '$InstanceName' ist bereits installiert!" "WARNING"
        Write-Log "Installation wird uebersprungen" "INFO"
        exit 0
    }
    
    $setupPath = Find-SqlServerSetup $LocalSetupPath
    if (-not $setupPath) {
        Write-Log "FEHLER: SQL Server Setup nicht gefunden!" "ERROR"
        exit 1
    }
    
    $installSuccess = Install-SqlServer -SetupPath $setupPath -InstanceName $InstanceName -SAPassword $SAPassword -InstallPath $InstallPath
    
    if (-not $installSuccess) {
        Write-Log "FEHLER: SQL Server Installation fehlgeschlagen!" "ERROR"
        exit 1
    }
    
    if (-not (Test-InstallationSuccess $InstanceName)) {
        Write-Log "FEHLER: Installation-Validierung fehlgeschlagen!" "ERROR"
        exit 1
    }
    
    $connectionString = Get-ConnectionString -InstanceName $InstanceName -UseEncryption $false
    Write-Log "Connection String (ohne Verschluesselung): $connectionString" "INFO"
    
    Write-Log "" "INFO"
    Write-Log "=== INSTALLATION ERFOLGREICH ABGESCHLOSSEN ===" "SUCCESS"
    Write-Log "Instanz: $InstanceName" "SUCCESS"
    Write-Log "Status: Bereit fuer Verschluesselungskonfiguration durch C# Anwendung" "SUCCESS"
    Write-Log "" "INFO"
    Write-Log "Naechste Schritte:" "INFO"
    Write-Log "1. Verschluesselung wird automatisch durch C# konfiguriert" "INFO"
    Write-Log "2. Firewall wird automatisch durch C# konfiguriert" "INFO"
    Write-Log "3. Verbindungstest wird durch C# durchgefuehrt" "INFO"
    
    exit 0
}
catch {
    Write-Log "KRITISCHER FEHLER: $($_.Exception.Message)" "ERROR"
    exit 1
}